<?php

namespace App\Models;

use App\Constants\Status;
use App\Traits\UserNotify;
use App\Traits\HasPermissions;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;


class User extends Authenticatable
{
    use HasApiTokens, UserNotify, HasPermissions;
    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'address' => 'object',
            'kyc_data' => 'object',
            'social_link' => 'object',
            'ver_code_send_at' => 'datetime'
        ];
    }


    public function loginLogs()
    {
        return $this->hasMany(UserLogin::class);
    }

    public function posts()
    {
        return $this->hasMany(Post::class);
    }

    public function experience()
    {
        return $this->hasMany(Experience::class);
    }

    public function all_post_comments_count()
    {
        $posts = $this->posts;
        $count = 0;
        foreach ($posts  as  $post) {
            $count +=  $post->comments->count();
        }
        return $count;
    }

    public function total_topic()
    {
        return $this->posts->pluck('category_id')->unique()->count();
    }

    public function transactions()
    {
        return $this->hasMany(Transaction::class)->orderBy('id','desc');
    }

    public function deposits()
    {
        return $this->hasMany(Deposit::class)->where('status','!=', Status::PAYMENT_INITIATE);
    }

    public function withdrawals()
    {
        return $this->hasMany(Withdrawal::class)->where('status','!=', Status::PAYMENT_INITIATE);
    }

    public function fullname(): Attribute {
        return new Attribute(
            get: fn() => $this->firstname || $this->lastname ? $this->firstname . ' ' . $this->lastname : '@'.$this->username,
        );
    }

    // SCOPES
    public function scopeActive()
    {
        return $this->where('status', Status::USER_ACTIVE);
    }

    public function scopeBanned()
    {
        return $this->where('status', Status::USER_BAN);
    }

    public function scopeEmailUnverified()
    {
        return $this->where('ev', Status::UNVERIFIED);
    }

    public function scopeMobileUnverified()
    {
        return $this->where('sv', Status::UNVERIFIED);
    }

    public function scopeKycUnverified()
    {
        return $this->where('kv', Status::UNVERIFIED);
    }

    public function scopeKycPending()
    {
        return $this->where('kv', Status::KYC_PENDING);
    }

    public function scopeKycVerified()
    {
        return $this->where('kv', Status::KYC_VERIFIED);
    }

    public function scopeEmailVerified()
    {
        return $this->where('ev', Status::VERIFIED);
    }

    public function scopeMobileVerified()
    {
        return $this->where('sv', Status::VERIFIED);
    }

    public function scopeWithBalance()
    {
        return $this->where('balance','>', 0);
    }


}
